<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AdminModel;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Permission;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Spatie\Permission\Models\Role;
use Stichoza\GoogleTranslate\GoogleTranslate;
use Illuminate\Support\Facades\Log;

class RoleController extends Controller
{
    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            if (auth()->user() && auth()->user()->id != 1) {
                if (in_array($request->route()->getActionMethod(), ['index', 'create', 'store', 'edit', 'update', 'destroy'])) {
                    abort(403, "You are not allowed to access this page.");
                }
            }
            return $next($request);
        });
    }
    public function index()
    {

        $roles = Role::where('name', '!=', 'superadmin')->orderBy('name', 'asc')->get();
        return view('admin.roles.index', [
            'roles' => $roles
        ]);
    }
    //create roles
    public function create()
    {
        $tr = new GoogleTranslate();
        $tr->setTarget(app()->getLocale());
        //echo $tr->translate('hello');
        //die();
        $permissions = Permission::orderBy('module', 'ASC')->get()->groupBy('module');
        $translatedPermissions = [];
        foreach ($permissions as $module => $modulePermissions) {
            foreach ($modulePermissions as $permission) {
                $permission->value = $permission->name;
                $permission->name = $tr->translate($permission->name);
                $translatedPermissions[$module][] = $permission;
            }
        }
        return view('admin.roles.create', compact('translatedPermissions'));
    }
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            $validations =  Validator::make($request->all(), [
                'name' => [
                    'required',
                    'min:3',
                    'unique:roles,name',
                ],
                'permissions' => [
                    'required',
                    'array',
                    'min:1',
                ],

            ]);

            if ($validations->fails()) {
                return jsonResponse(false, 400, $validations->errors());
            }
            $permissions = $request->input('permissions', []);
            $additionalPermissions = [];
            foreach ($permissions as $permission) {
                $module = Permission::where('name', $permission)->first();
                $mod = ($module) ? $module->module : '';
                $listPermission = "list - $mod";
                if (str_contains($permission, 'edit') || str_contains($permission, 'delete')  || str_contains($permission, 'conversation')) {
                    if (str_contains($permission, 'conversation')) {
                        $viewPermission = "view - $mod";

                        if (!in_array($listPermission, $permissions) && !in_array($viewPermission, $permissions)) {
                            return jsonResponse(false, 400, ['permissions' => [__('message.module_permission_list_view_required', ['mod' => $mod])]]);
                        }
                    } else {
                        if (!in_array($listPermission, $permissions)) {
                            return jsonResponse(false, 400, ['permissions' => [__('message.module_permission_list_required', ['mod' => $mod])]]);
                        }
                    }
                }

                if ($permission == 'view - deal detail') {
                    $requiredPermission = 'list - ongoing deals';
                    if (!in_array($requiredPermission, $permissions)) {
                        $additionalPermissions[] = $requiredPermission;
                    }
                }
                if (str_contains($permission, 'deal detail')) {
                    if (!in_array('view - deal detail', $permissions)) {
                        return jsonResponse(false, 400, ['permissions' => [__('message.module_permission_view_required', ['mod' => $mod])]]);
                    }
                }
            }

            $permissions =  array_unique(array_merge($permissions, $additionalPermissions));

            $role =  Role::create(['name' => $request->name]);
            //assign permissions one by one

            if (!empty($request->permissions)) {
                $role->syncPermissions($permissions);
            }

            DB::commit();
            $message = __('message.role_added_successfully');
            $extra = ['redirect' => route('admin.role.index')];
            return jsonResponse(true, 200, $message, $extra);
        } catch (\Exception $e) {
            DB::rollBack();
            return jsonResponse(false, 500, __('message.internal_server_error') . $e->getMessage());
        }
    }
    public function edit($id)
    {

        $tr = new GoogleTranslate();
        $tr->setTarget(app()->getLocale());
        $role = Role::findOrFail($id);
        if ($role->name === "superadmin") {
            return redirect()->route('admin.role.index')->with('error', __('message.role_edit_not_allowed'));
        }
        $hasPermissions = $role->permissions->pluck('name');
        $permissions = Permission::orderBy('module', 'ASC')->get()->groupBy('module');
        $translatedPermissions = [];
        foreach ($permissions as $module => $modulePermissions) {
            foreach ($modulePermissions as $permission) {
                $permission->value = $permission->name;
                $permission->name = $tr->translate($permission->name);
                $translatedPermissions[$module][] = $permission;
            }
        }

        return view('admin.roles.edit', [
            'translatedPermissions' => $translatedPermissions,
            'hasPermissions' => $hasPermissions,
            'role' => $role,
        ]);
    }
    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        try {

            $role = Role::findOrFail($id);
            if ($role->name === "superadmin") {
                $extra = ['redirect' => route('admin.role.index')];
                return jsonResponse(false, 500, __('message.role_delete_not_allowed'), $extra);
            }
            if (empty($role)) {
                $message = __('message.invalid_role');
                $extra = ['redirect' =>  route('admin.role.index')];
                return jsonResponse(false, 500, $message, $extra);
            }
            $validations =  Validator::make($request->all(), [
                'name' => 'required|min:3|unique:roles,name,' . $id,
                'permissions' => [
                    'required',
                    'array',
                    'min:1',
                ],
            ]);

            if ($validations->fails()) {
                return jsonResponse(false, 400, $validations->errors());
            }
            $permissions = $request->input('permissions', []);
            $additionalPermissions = [];
            foreach ($permissions as $permission) {
                $module = Permission::where('name', $permission)->first();
                $mod = ($module) ? $module->module : '';
                $listPermission = "list - $mod";
                if (str_contains($permission, 'edit') || str_contains($permission, 'delete')  || str_contains($permission, 'conversation')) {
                    if (str_contains($permission, 'conversation')) {
                        $viewPermission = "view - $mod";

                        if (!in_array($listPermission, $permissions) && !in_array($viewPermission, $permissions)) {
                            return jsonResponse(false, 400, ['permissions' => [__('message.module_permission_list_view_required', ['mod' => $mod])]]);
                        }
                    } else {
                        if (!in_array($listPermission, $permissions)) {
                            return jsonResponse(false, 400, ['permissions' => [__('message.module_permission_list_required', ['mod' => $mod])]]);
                        }
                    }
                }
                if ($permission == 'view - deal detail') {
                    $requiredPermission = 'list - ongoing deals';
                    if (!in_array($requiredPermission, $permissions)) {
                        $additionalPermissions[] = $requiredPermission;
                    }
                }
                if (str_contains($permission, 'deal detail')) {
                    if (!in_array('view - deal detail', $permissions)) {
                        return jsonResponse(false, 400, ['permissions' => [__('message.module_permission_view_required', ['mod' => $mod])]]);
                    }
                }
            }
            $permissions = array_unique(array_merge($permissions, $additionalPermissions));
            $role->name = $request->name;
            $role->save();

            if (!empty($permissions)) {
                $role->syncPermissions($permissions);
            } else {
                $role->syncPermissions([]);
            }
            DB::commit();
            $message = __('message.role_updated_successfully');
            $extra = ['redirect' => route('admin.role.index')];
            return jsonResponse(true, 200, $message, $extra);
        } catch (\Exception $e) {
            DB::rollBack();
            return jsonResponse(false, 500, __('message.internal_server_error') . $e->getMessage());
        }
    }
    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $role = Role::find($id);
            if (empty($role)) {
                $message = __('message.invalid_role');
                $extra = ['redirect' =>  route('admin.role.index')];
                return jsonResponse(false, 500, $message, $extra);
            }
            if ($role->name === "superadmin") {
                $extra = ['redirect' => route('admin.role.index')];
                return jsonResponse(false, 500, __('message.role_delete_not_allowed'), $extra);
            }
            $role->delete();
            DB::commit();
            $message = __('message.role_deleted_successfully');
            $extra = ['redirect' => route('admin.role.index')];
            return jsonResponse(true, 200, $message, $extra);
        } catch (\Exception $e) {
            //dd('here');
            DB::rollBack();
            //Log::error('Role Deletion Error: ' . $e->getMessage());
            return jsonResponse(false, 500, $e->getMessage());
        }
    }
}
