<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\profileModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

class ProfileController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:update profile - setting')->only(['profile', 'profileUpdate']);
        $this->middleware('permission:change password - setting')->only(['changePassword', 'changePasswordProcess']);
    }
    public function profile()
    {
        $admin = Auth::user();
        return view('admin.default.profile', compact('admin'));
    }
    public function profileUpdate(Request $request, $id)
    {
        try {
            DB::beginTransaction();
            $admin = profileModel::find($id);
            if (!$admin) {
                // abort(404);
                $message = __('message.invalid_details');
                return jsonResponse(false, 500,  $message);
            }
            $validator = (new ProfileModel)->profileValidation($request->all(), $admin->id);
            if ($validator->fails()) {
                return jsonResponse(false, 400, $validator->errors());
            }
            $existMobileNumber = (new ProfileModel)->existMobileNumber($request->dial_code, $request->dial_code_iso, $request->mobile_number, $admin->id);
            if (!empty($existMobileNumber)) {
                $message = __('message.mobile_number_already_exists');
                return jsonResponse(false, 207, $message);
            }
            if (!isValidNumber($request->dial_code, $request->dial_code_iso, $request->mobile_number)) {
                $message = __('message.invalid_mobile_number');
                return jsonResponse(false, 207, $message);
            }
            $data = [
                'name'          => $request->name,
                'email'         => $request->email,
                'mobile_number'  => $request->mobile_number,
                'dial_code'     => $request->dial_code,
                'dial_code_iso' => $request->dial_code_iso,
            ];
            $data = $request->except(['image']);

            if ($request->hasFile('image')) {
                $image =  uploadFile('image', ADMIN_PATH, $admin->image);
                $data['image'] = $image;
            }
            $admin->update($data);
            DB::commit();

            $message = __('message.profile_updated_successfully');
            $extra = ['redirect' => route('admin.profile')];
            return jsonResponse(true, 200, $message, $extra);
        } catch (\Exception $e) {
            DB::rollback();
            $message = __('message.internal_server_error') . $e->getMessage();
            return jsonResponse(false, 500, $message);
        }
    }
    public function changePassword()
    {
        $admin = Auth::user();
        return view('admin.default.change-password', compact('admin'));
    }
    public function changePasswordProcess(Request $request, $id)
    {
        try {
            DB::beginTransaction();
            $admin = profileModel::find($id);
            $validator = (new ProfileModel)->changePasswordValidation($request->all(), $admin->id);
            if ($validator->fails()) {
                return jsonResponse(false, 400, $validator->errors());
            }

            if (!Hash::check($request->old_password, $admin->password)) {

                $message = __('message.current_password_incorrect');
                return jsonResponse(false, 207, $message);
            }
            if ($request->new_password !== $request->confirm_password) {

                $message = __('message.password_mismatch');
                return jsonResponse(false, 207, $message);
            }
            //update
            $admin->password =  Hash::make($request->new_password);
            $admin->save();
            DB::commit();
            $message = __('message.password_changed_successfully');
            $extra = ['redirect' => route('admin.profile')];
            return jsonResponse(true, 200, $message, $extra);
        } catch (\Exception $e) {
            DB::rollback();
            $message = __('message.internal_server_error') . $e->getMessage();
            return jsonResponse(false, 500, $message);
        }
    }
}
