<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\AdminModel;
use Illuminate\Http\Request;
use Spatie\Permission\Models\Role;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;

class UserController extends Controller
{
    /**
     * Display a listing of the resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function __construct()
    {
        $this->middleware(function ($request, $next) {
            if (auth()->user() && auth()->user()->id != 1) {
                if (in_array($request->route()->getActionMethod(), ['index', 'create', 'store', 'edit', 'update', 'destroy'])) {
                    abort(403, "You are not allowed to access this page.");
                }
            }
            return $next($request);
        });
    }
    public function index()
    {
        $admins = AdminModel::orderBy('created_at', 'desc')->where('id', '!=', 1)->get();
        return view('admin.users.index', [
            'admins' => $admins
        ]);
    }

    /**
     * Show the form for creating a new resource.
     *
     * @return \Illuminate\Http\Response
     */
    public function create()
    {
        $roles = Role::where('id', '!=', 20)->orderBy('name', 'asc')->get();
        return view('admin.users.create', [
            'roles' => $roles
        ]);
    }

    /**
     * Store a newly created resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @return \Illuminate\Http\Response
     */
    public function store(Request $request)
    {
        try {
            DB::beginTransaction();
            $validations =  Validator::make($request->all(), [
                'name' => 'required|string|max:30|min:3',
                'mobile_number' => 'required',
                'email' => 'required|email|unique:admins,email',
                'image' => 'nullable|image|mimes:jpg,jpeg,webp,png,gif|max:2048',
                'role' => 'required|exists:roles,name',
                'password' => [
                    'required',
                    'regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[\W_]).{8,}$/',
                    'min:8',
                    'confirmed',
                ],
                'password_confirmation' => 'required',

            ]);
            $image =  "default-admin.png";
            if ($validations->fails()) {
                return jsonResponse(false, 400, $validations->errors());
            }
            $existMobileNumber = (new AdminModel())->existMobileNumber($request->dial_code, $request->dial_code_iso, $request->mobile_number);
            if (!empty($existMobileNumber)) {
                $message = __('message.mobile_number_already_exists');
                return jsonResponse(false, 207, $message);
            }
            if (!isValidNumber($request->dial_code, $request->dial_code_iso, $request->mobile_number)) {
                $message = __('message.invalid_mobile_number');
                return jsonResponse(false, 207, $message);
            }
            if ($request->hasFile('image')) {
                $image =  uploadFile('image', ADMIN_PATH);
            }
            $admin =  AdminModel::create([
                'name' => $request->name,
                'email' => $request->email,
                'mobile_number' => $request->mobile_number,
                'dial_code' => $request->dial_code,
                'dial_code_iso' => $request->dial_code_iso,
                'password' =>  Hash::make($request->password),
                'image' =>  $image,
                'status' =>  $request->status,

            ]);


            $admin->syncRoles($request->role);

            DB::commit();
            $message = __('message.user_added_successfully');
            $extra = ['redirect' => route('admin.user.index')];
            return jsonResponse(true, 200, $message, $extra);
        } catch (\Exception $e) {
            DB::rollBack();
            return jsonResponse(false, 500, __('message.internal_server_error') . $e->getMessage());
        }
    }

    /**
     * Display the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function show($id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function edit($id)
    {
        $admin = AdminModel::find($id);
        $roles = Role::where('id', '!=', 20)->orderBy('name', 'asc')->get();
        $adminRole = $admin->roles->pluck('id');
        //for single role get - if admin assign only single role
        //$adminRole = $admin->roles->first()->id ?? null;
        return view('admin.users.edit', [
            'admin' => $admin,
            'roles' => $roles,
            'adminRole' => $adminRole,

        ]);
    }

    /**
     * Update the specified resource in storage.
     *
     * @param  \Illuminate\Http\Request  $request
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function update(Request $request, $id)
    {
        try {
            DB::beginTransaction();
            $admin = AdminModel::find($id);
            if (empty($admin)) {
                $message = __('message.invalid_user');
                $extra = ['redirect' =>  route('admin.user.index')];
                return jsonResponse(false, 500, $message, $extra);
            }
            $rules = [
                'name' => 'required|string|max:30|min:3',
                'mobile_number' => 'required',
                'email' => 'required|email|unique:admins,email,' . $id . 'id',
                'image' => 'nullable|image|mimes:jpg,jpeg,webp,png,gif|max:2048',
                'role' => 'required|exists:roles,name',
            ];
            if ($request->filled('password')) {
                $rules['password'] = [
                    'required',
                    'string',
                    'min:8',
                    'regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[\W_]).{8,}$/',
                    'confirmed'

                ];
            }
            $validations =  Validator::make($request->all(), $rules);
            $existMobileNumber = (new AdminModel())->existMobileNumber($request->dial_code, $request->dial_code_iso, $request->mobile_number, $admin->id);
            if (!empty($existMobileNumber)) {
                $message = __('message.mobile_number_already_exists');
                return jsonResponse(false, 207, $message);
            }

            if (!isValidNumber($request->dial_code, $request->dial_code_iso, $request->mobile_number)) {
                $message = __('message.invalid_mobile_number');
                return jsonResponse(false, 207, $message);
            }
            $image =  $admin->image;
            if ($validations->fails()) {
                return jsonResponse(false, 400, $validations->errors());
            }


            if ($request->hasFile('image')) {
                $image =  uploadFile('image', ADMIN_PATH, $admin->image);
            }
            //password check if null
            if (!empty($request->password)) {
                $password = Hash::make($request->password);
            } else {
                $password = $admin->password;
            }
            $admin->name = $request->name;
            $admin->email = $request->email;
            $admin->mobile_number = $request->mobile_number;
            $admin->dial_code = $request->dial_code;
            $admin->dial_code_iso = $request->dial_code_iso;
            $admin->image = $image;
            $admin->password = $password;
            $admin->status = $request->status;
            $admin->save();
            $admin->syncRoles($request->role);

            DB::commit();
            $message = __('message.user_updated_successfully');
            $extra = ['redirect' => route('admin.user.index')];
            return jsonResponse(true, 200, $message, $extra);
        } catch (\Exception $e) {
            DB::rollBack();
            return jsonResponse(false, 500, __('message.internal_server_error') . $e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     *
     * @param  int  $id
     * @return \Illuminate\Http\Response
     */
    public function destroy($id)
    {
        try {
            $admin = AdminModel::find($id);
            if (empty($admin)) {
                $message = __('message.invalid_user_data');
                $extra = ['redirect' =>  route('admin.user.index')];
                return jsonResponse(false, 500, $message, $extra);
            }
            $admin->delete();
            DB::commit();
            $message = __('message.user_deleted_successfully');
            $extra = ['redirect' => route('admin.user.index')];
            return jsonResponse(true, 200, $message, $extra);
        } catch (\Exception $e) {
            DB::rollBack();
            return jsonResponse(false, 500, 'Internal server error' . $e->getMessage());
        }
    }
}
