<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\CustomerRankModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;



class CustomerRankController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:list - customer rank')->only(['index']);
        $this->middleware('permission:edit - customer rank')->only(['edit', 'update']);
    }
    public function index()
    {
        $ranks = CustomerRankModel::orderBy('min_successful_deals', 'asc')->get();
        return view('admin.customer-ranks.index', compact('ranks'));
    }
    public function edit($id)
    {
        $rank = CustomerRankModel::findOrFail($id);
        return view('admin.customer-ranks.edit', compact('rank'));
    }
    public function update(Request $request, $id)
    {
        DB::beginTransaction();
        try {
            $rank = CustomerRankModel::findOrFail($id);
            $allowedExtensions = implode(',', imageExtensions());

            $validation = Validator::make($request->all(), [
                'title_en' => 'required|string|max:100|min:3',
                'title_es' => 'required|string|max:100|min:3',
                'min_successful_deals' => [
                    'required',
                    'integer',
                    'min:1',
                    Rule::unique('customer_ranks')->ignore($id),
                    function ($attribute, $value, $fail) use ($id) {
                        $ranks = CustomerRankModel::orderBy('min_successful_deals')->get();
                        $currentIndex = $ranks->search(fn($r) => $r->id == $id);
                        $prev = $ranks->get($currentIndex - 1);
                        $next = $ranks->get($currentIndex + 1);
                        if ($prev && $value <= $prev->min_successful_deals) {
                            $fail(__('message.min_successful_deal_less_validation', ['prev' => $prev->min_successful_deals]));
                        }
                        if ($next && $value >= $next->min_successful_deals) {
                            $fail(__('message.min_successful_deal_greater_validation', ['next' => $next->min_successful_deals]));
                        }
                    }
                ],
                'icon_type' => 'required|in:star,medal,image', // adjust to match rankFilter() keys
                'icon_count' => 'required_if:icon_type,star,medal|integer|min:1|max:5',
                'file' => 'nullable|image|mimes:' . $allowedExtensions . '|max:500'
            ]);

            if ($validation->fails()) {
                return jsonResponse(false, 400, $validation->errors());
            }
            if (($request->icon_type == MEDAL || $rank->icon_type == MEDAL) && $request->icon_count != 1) {
                return jsonResponse(false, 400, ['icon_count' => [__('message.medal_count_validation')]]);
            }
            $icon = $rank->icon;
            if ($request->has('file')) {
                $icon =    uploadFile('file', RANK_ICONS, $rank->icon);
            }
            //upload requestes
            $rank->title_en = $request->title_en;
            $rank->title_es = $request->title_es;
            $rank->min_successful_deals = $request->min_successful_deals;
            $rank->icon_type = $request->icon_type;
            $rank->icon_count = $request->icon_count;
            $rank->icon = $icon; //update icon
            $rank->save();
            DB::commit();
            $extra = ['redirect' => route('admin.customerRank.index')];
            return jsonResponse(true, 200, __('message.rank_updated_successfully'), $extra);
        } catch (\Exception $e) {
            DB::rollBack();
            return jsonResponse(false, 500, __('message.internal_server_error') . $e->getMessage());
        }
    }
}
