<?php

namespace App\Models;

use Illuminate\Database\Eloquent\Factories\HasFactory;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Database\Eloquent\SoftDeletes;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Laravel\Sanctum\HasApiTokens;
use Illuminate\Foundation\Auth\User as Authenticatable;
use Illuminate\Validation\Rule;
use Spatie\Permission\Traits\HasRoles;
use PDO;

class CustomerModel extends Authenticatable
{
    use SoftDeletes;
    use  HasApiTokens, HasFactory, HasRoles;
    protected $dates = ['deleted_at'];
    protected $table = 'customers';
    protected $fillable = [
        'full_name',
        'email',
        'user_name',
        'mobile_number',
        'dial_code',
        'dial_code_iso',
        'number_verified_at',
        'email_verified_at',
        'password',
        'token',
        'address',
        'created_at',
        'updated_at',
        'status',
        'description',
        'image',
        'email_otp',
        'is_email_verified',
        'is_number_verified',
        'account_holder_name',
        'account_number',
        'bank_name',
        'country_id',
        'city_id',
        'zip_code',
        'state_id',
        'is_profile_updated',
        'fcm_token',
        'language_code',
        'successful_deals_count',
        'bad_deals_count',
        'bad_deals_after_activation',
        'last_inactivated_at',
        'last_reactivated_at',
        'terms_conditions',
        'is_invited',
        'rank_id',
        'id_front_side',
        'id_back_side'
    ];
    protected $casts = [
        'is_number_verified' => 'boolean',
        'is_invited' => 'boolean',
    ];
    public function pagination(Request $request)
    {
        $filter = 1;
        $perPage = 10;
        $sortOrder = 'desc';
        $sortEntity = 'customers.id';


        $fields = "
            customers.*
        ";

        if ($request->has('per_page') && $request->get('per_page') != '') {
            $perPage = $request->get('per_page');
        }



        if ($request->has('keyword') && $request->get('keyword') != '') {
            $filter .= " and (
                customers.full_name like '%" . addslashes($request->get('keyword')) . "%'
                or customers.mobile_number like '%" . addslashes($request->get('keyword')) . "%'
                or customers.dial_code like '%" . addslashes($request->get('keyword')) . "%'
                or customers.status like '%" . addslashes($request->get('keyword')) . "%')";
        }

        if ($request->has('status') && $request->get('status') != '') {
            if ($request->get('status') != 'delete') {
                $filter .= " and customers.deleted_at IS NULL";
                $filter .= " and customers.status = '" . addslashes($request->get('status')) . "'";
            } else {
                $filter .= " and customers.deleted_at IS NOT NULL ";
            }
        }

        if ($request->has('sort_by') && $request->get('sort_by') != '') {
            $sortEntity = $request->get('sort_by');
        }

        if ($request->has('sort_order') && $request->get('sort_order') != '') {
            $sortOrder = $request->get('sort_order');
        }

        return $this
            ->select(DB::raw($fields))
            ->whereRaw($filter)
            ->where('is_number_verified', 1)
            ->where('id', '!=', '1')
            ->orderBy($sortEntity, $sortOrder)
            ->paginate($perPage);
    }

    //customer add/edit validations
    public function customerValidation($inputs = [], $id = null)
    {
        $rules = [
            'full_name' => 'required|string|max:30|min:3',
            'mobile_number' => 'required',
            /*'email' => [
                'required',
                'email:rfc,dns',
                Rule::unique('customers')->whereNull('deleted_at')
            ],*/
            'dial_code' => 'required',
            'dial_code_iso' => 'required',
            'status' => 'required|in:active,inactive',
            'image' => 'nullable|image|mimes:jpg,jpeg,webp,png,gif|max:2048',
        ];

        if ($id) {
            /*$rules['email'] = [
                'required',
                'email:rfc,dns',
                Rule::unique('customers')->ignore($id)->whereNull('deleted_at')
            ];*/
            $rules['password'] = 'nullable';
        }
        return Validator($inputs, $rules);
    }

    //mobile number exist
    public function existMobileNumber($dial_code, $dial_code_iso, $mobile_number, $skip_id = null)
    {
        $result = $this
            ->where('dial_code', $dial_code)
            ->where('dial_code_iso', $dial_code_iso)
            ->where('mobile_number', $mobile_number);

        //edit case
        if ($skip_id) {
            $result = $result->where('id', '!=', $skip_id);
        }

        return $result->first();
    }

    public function signupApiValidation($requests)
    {
        $rules = [
            'dial_code' => 'required',
            'dial_code_iso' => 'required',
            'full_name' => 'required|min:3',
            /*'email' => [
                'required',
                'email',
                // Rule::unique('customers')->whereNull('deleted_at')
            ],*/
            'mobile_number' => 'required|numeric',
            'address' => 'required',
            'password' =>
            [
                'required',
                'min:8',
                'string',
                'regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d)(?=.*[@$!%*?&])[A-Za-z\d@$!%*?&]{8,}$/'
            ],
            'terms_conditions' => 'required|boolean',
            'id_front_side' => 'required|image|mimes:jpg,jpeg,webp,png,gif',
            'id_back_side' => 'required|image|mimes:jpg,jpeg,webp,png,gif',
            'language_code' => 'required|in:en,es'

        ];
        $messages =
            [
                'password.regex' => 'The password must be at least 8 characters long and include at least one uppercase letter, one lowercase letter, one number, and one special character.'
            ];
        return Validator($requests, $rules, $messages);
    }
    //existotp
    /*public function OtpExist($email, $otp)
    {
        return $this->where('email_otp', $otp)->where('email', $email)->first();
    }
    public function getCustomerData($email)
    {
        return $this->where('email', $email)->where('status', 'active')->first();
    }
    public function forgotOtpCheck($email, $otp)
    {
        return $this->where('email', $email)->where('email_otp', $otp)->first();
    }*/
    //save customer rank id
    public function updateRank()
    {
        $ranks = CustomerRankModel::orderBy('min_successful_deals', 'asc')->get();

        foreach ($ranks as $rank) {

            if ($this->successful_deals_count >= $rank->min_successful_deals) {
                $this->rank_id = $rank->id;
                $this->save();
                break;
            }
        }
    }
    public function rank()
    {
        return $this->belongsTo(CustomerRankModel::class, 'rank_id'); // 'customer_id' is the foreign key in the CustomerRank table
    }
    public function getGoodDealsRank()
    {
        $rank = $this->rank;
        $lang = $this->language_code;
        return [
            'icon_type' => $rank?->icon_type,
            'icon_count' => $rank?->icon_count,
            'title' => $rank?->getTitleByLang($lang),
            'icon' => $rank?->icon ? asset(RANK_ICONS . '/' . $rank->icon) : null,
        ];
    }
    public function totalDealCount()
    {
        return DealModel::where('from_customer_id', $this->id)->orWhere('to_customer_id', $this->id)->count();
    }
    public function isDeleted(): bool
    {
        return $this->deleted_at !== null;
    }
}
