<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use App\Models\BannerModel;
use App\Models\CustomerModel;
use App\Models\DealModel;
use App\Models\NotificationModel;
use Illuminate\Support\Facades\Auth as CustomerAuth;
use Laravel\Sanctum\PersonalAccessToken;
use Illuminate\Support\Facades\Validator;

class HomeController extends Controller
{
    //get profile of customer
    public function getProfile()
    {
        try {
            DB::beginTransaction();
            $customer =  CustomerAuth::user();

            $countryName = DB::table('countries')->where('id', $customer->country_id)->value('name');
            $stateName = DB::table('states')->where('id', $customer->state_id)->value('name');
            $cityName = DB::table('cities')->where('id', $customer->city_id)->value('name');
            DB::commit();
            $data = [

                'customer_id' => $customer->id,
                'full_name' => $customer->full_name,
                'dial_code' => $customer->dial_code,
                'dial_code_iso' => $customer->dial_code_iso,
                'mobile_number' => $customer->mobile_number,
                'image' => (string)getUserImage($customer->image, USER_UPLOADED_IMG_PATH),
                'address' => $customer->address,
                'country_id' => $customer->country_id,
                'country_name' => $countryName,
                'state_id' => $customer->state_id,
                'state_name' => $stateName,
                'city_id' => $customer->city_id,
                'city_name' => $cityName,
                'zip_code' => $customer->zip_code,
                'account_holder_name' => $customer->account_holder_name,
                'account_number' => $customer->account_number,
                'bank_name' => $customer->bank_name,
                'is_profile_updated' => $customer->is_profile_updated,
                'id_back_side' => asset(CUSTOMER_PATH . '/' . $customer->id_back_side),
                'id_front_side' => asset(CUSTOMER_PATH . '/' . $customer->id_front_side)

            ];
            $message = __('message.data_found');
            return apiResponse(200, $message, $data);
        } catch (\Exception $e) {
            DB::rollback();
            return getErrorResponse($e);
        }
    }
    public function updateProfileImage(Request $request)
    {
        try {
            DB::beginTransaction();
            $customer =  CustomerAuth::user();

            if (!$customer) {
                $message = __('message.not_found');
                return apiResponse(400, $message);
            }
            $validation = Validator::make($request->all(), [
                'image' => 'required|image|mimes:jpeg,png,jpg,gif,webp',
                // 'image' => 'required|image|mimes:jpeg,png,jpg,gif,webp|max:2048'
            ]);
            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }
            $image = $customer->imge;
            //$inputs = $request->except(['image']);
            if ($request->hasFile('image')) {
                $image =  uploadFile('image', CUSTOMER_PATH, $customer->image);
            }
            CustomerModel::where('id', $customer->id)->update(['image' => $image]);
            $customer = CustomerModel::find($customer->id);
            DB::commit();



            return apiResponse(200, __('message.image_updated_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }

    public function updateBankDetails(Request $request)
    {
        try {
            DB::beginTransaction();
            $customer =  CustomerAuth::user();
            if (!$customer) {
                $message = __('message.not_found');
                return apiResponse(400, $message);
            }
            $validation = Validator::make($request->all(), [
                'account_holder_name' => 'required|string|min:3|max:100|regex:/^[A-Za-z\s\.\-]+$/',
                'account_number' => 'required|regex:/^[A-Za-z0-9]+$/|min:6|max:34',
                'bank_name' => 'required|string|min:3|max:100|regex:/^(?!\d+$)[A-Za-z0-9\s\.\-]+$/'
            ]);
            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }

            CustomerModel::where('id', $customer->id)->update([
                'account_holder_name' => $request->account_holder_name,
                'account_number' => $request->account_number,
                'bank_name' => $request->bank_name,
            ]);
            $customer = CustomerModel::find($customer->id);
            checkProfileUpdate($customer);
            DB::commit();


            return apiResponse(200, __('message.data_updated_successfully'));
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
    public function updateProfile(Request $request)
    {
        try {
            DB::beginTransaction();
            $customer =  CustomerAuth::user();
            if (!$customer) {
                $message = __('message.not_found');
                return apiResponse(400, $message);
            }
            $validation = Validator::make($request->all(), [
                'full_name' => 'required|min:3',
                'address' => 'required|min:3|max:500',
                'zip_code' => 'required|min:3|max:10|regex:/^[A-Za-z0-9\s\-]+$/',
                'country_id' => 'required|exists:countries,id',
                'state_id' => 'exists:states,id',
                'city_id' => 'exists:cities,id',

            ]);
            if ($validation->fails()) {
                return apiResponse(400, $validation->errors()->first());
            }

            CustomerModel::where('id', $customer->id)->update([
                'full_name' => $request->full_name,
                'address' => $request->address,
                'zip_code' => $request->zip_code,
                'country_id' => $request->country_id,
                'state_id' => $request->state_id,
                'city_id' => $request->city_id,
            ]);
            $customer = CustomerModel::find($customer->id);
            checkProfileUpdate($customer);
            DB::commit();
            return apiResponse(200, __('message.data_updated_success'));
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
    //dashboard api
    public function dashboard(Request $request)
    {
        try {
            db::beginTransaction();
            //fecth banner api
            $banners = BannerModel::where('status', 'active')
                ->orderBy('created_at', 'desc')
                ->limit(3)
                ->get()
                ->map(function ($banner) {
                    $banner->image = asset(BANNER_PATH . '/' . $banner->image);
                    unset($banner->video_poster);
                    return $banner;
                });
            $customerId = getLoggedInCustomerId();
            $recentDeals = DealModel::with(['fromCustomer', 'toCustomer'])->where(function ($query) use ($customerId) {
                $query->where('from_customer_id', $customerId)->orWhere('to_customer_id', $customerId);
            })
                ->orderBy('updated_at', 'desc')
                ->limit(5)
                ->get();
            //get requests deal count which are pending by customer side
            $getRequestCount = DealModel::where('to_customer_id', $customerId)
                ->where('deal_status', PENDING)->count();

            $dealsWithSecondParty = $recentDeals->map(function ($deal) use ($customerId) {
                if ($deal->from_customer_id == $customerId) {
                    $secondPartyId = $deal->toCustomer;
                } else {
                    $secondPartyId = $deal->fromCustomer;
                }

                $dealType = dealTypeViceVersa($deal, $customerId) == 'Buy' ? __('message.buy') : __('message.sell');
                return [
                    'deal_id' => $deal->id,
                    'deal_amount' => $deal->deal_amount,
                    'deal_amount_in_format' => formatCurrency($deal->deal_amount, $deal->currency_code),
                    'deal_type' => $dealType,
                    'deal_status' => checkDisputeDeal($deal->deal_status),
                    'deal_status_lang_wise' => __('readableStatuses.' . checkDisputeDeal($deal->deal_status)),
                    'deal_next_step' => $deal->getNextStepIndicator($customerId),
                    'deal_number' => $deal->deal_number,
                    'second_party' => [
                        'id' => $secondPartyId->id,
                        'full_name' => $secondPartyId->full_name,
                        'image' => asset('uploads/customers/' . $secondPartyId->image),
                    ],
                ];
            });
            //fecth unread notification count
            $unreadNotificationCount = NotificationModel::where('customer_id', $customerId)
                ->where('is_read', 0)
                ->count();
            DB::commit();
            $data = [
                'banners' => $banners,
                'recent_deals' => $dealsWithSecondParty,
                'unread_notification_count' => $unreadNotificationCount,
                'get_request_count' => $getRequestCount,
            ];
            return apiResponse(200, __('message.dashboard_data_retrieved'), $data);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
    //get countries
    public function getCountries()
    {
        try {
            $countries = DB::table('countries')
                ->select('id', 'name')
                ->orderBy('name')
                ->get();
            return apiResponse(200, __('message.countries_data_retrieved_successfully'), $countries);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
    public function getStateByCountry($country_id)
    {
        try {
            $states = DB::table('states')
                ->where('country_id', $country_id)
                ->select('id', 'name')
                ->orderBy('name')->get();
            return apiResponse(200, __('message.states_data_retrieved_successfully'), $states);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
    public function getCitiesByState($id)
    {
        try {
            $cities = DB::table('cities')
                ->where('state_id', $id)
                ->select('id', 'name')
                ->orderBy('name')->get();
            return apiResponse(200,  __('message.cities_data_retrieved_successfully'), $cities);
        } catch (\Exception $e) {
            DB::rollBack();
            return getErrorResponse($e);
        }
    }
}
