<?php

namespace App\Http\Controllers\Website;

use App\Http\Controllers\Controller;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Mail;
use App\Mail\ContactMail;

class WebsiteController extends Controller
{
    //index
    public function index()
    {
        return view('website.index');
    }
    //about us
    public function about()
    {
        return view('website.about');
    }
    public function downloadApp()
    {
        return view('website.download-app');
    }
    public function contact()
    {
        return view('website.contact');
    }
    public function submitContact(Request $request)
    {
        try {
            $validator = Validator::make($request->all(), [
                'name' => 'required|string|max:100',
                'email' => 'required|email|max:150',
                'mobile_number' => 'required|regex:/^[0-9]{6,15}$/',
                'message' => 'nullable|string',
                'dial_code' => 'required|string|max:5',
                'dial_code_iso' => 'required|size:2|string',
            ]);
            if ($validator->fails()) {
                return jsonResponse(false, 400, $validator->errors());
            }
            // Validate mobile number
            if (!isValidNumber($request->dial_code, $request->dial_code_iso, $request->mobile_number)) {
                $message = __('message.invalid_mobile_number');
                return jsonResponse(false, 207, $message);
            }
            //submit mail
            $data = [
                'name'          => $request->name,
                'email'         => $request->email,
                'dial_code'     => $request->dial_code,
                'dial_code_iso' => $request->dial_code_iso,
                'mobile_number' => $request->mobile_number,
                'contact_message'       => $request->contact_message, // make sure it’s included
            ];
            $extra = [
                'redirect' => route('website.contact')
            ];
            try {
                // send verification email
                Mail::to(config('mail.from.address'))->send(new ContactMail($data));
                return jsonResponse(true, 200, __('message.contact_success_dialog'), $extra);
            } catch (\Exception $e) {
                if (str_contains($e->getMessage(), '550')) {
                    //return apiResponse(400, __('message.email_invalid'), $data);
                    return jsonResponse(false, 500, __('message.email_invalid'), $extra);
                }
                //return apiResponse(500, __('message.email_send_error'), $data);
                return jsonResponse(false, 500, $e->getMessage(), $extra);
            }
        } catch (\Exception $e) {
            $message = __('message.internal_server_error') . $e->getMessage();
            return jsonResponse(false, 500, $message);
        }
    }
    public function privacyPolicy()
    {
        return view('website.privacy-policy');
    }
    public function termsConditions()
    {
        return view('website.terms-conditions');
    }
}
