<?php

namespace App\Http\Controllers\Admin;

use App\Http\Controllers\Controller;
use App\Models\CustomerModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Validator;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\DB;
use App\Support\PaginationHelper;
use App\Services\SMSService;

class CustomerController extends Controller
{
    public function __construct()
    {
        $this->middleware('permission:list - customer')->only('index');
        $this->middleware('permission:edit - customer')->only(['edit', 'update']);
        $this->middleware('permission:delete - customer')->only('destroy');
    }
    public function index(Request $request)
    {
        $customers = (new CustomerModel)->pagination($request);
        if ($request->ajax()) {
            return response()->json([
                'data' => view('admin.customer.pagination', compact('customers'))->render()
            ]);
        }

        return view('admin.customer.index', compact('customers'));
    }
    public function edit($id)
    {
        $customer = CustomerModel::with('rank')->findOrFail($id);
        $countries = db::table('countries')->select('id', 'name')->get();
        $totalDeals = $customer->totalDealCount();
        return view('admin.customer.edit', compact('customer', 'countries', 'totalDeals'));
    }
    public function update(Request $request, $id)
    {

        try {
            DB::beginTransaction();

            $inputs = $request->all();

            $customer = (new CustomerModel)->find($id);
            if (empty($customer)) {
                $message = __('message.invalid_details');
                $extra = ['redirect' =>  route('admin.customer.index')];
                return jsonResponse(false, 500, $message, $extra);
            }

            $validator = (new CustomerModel)->customerValidation($inputs, $customer->id);
            if ($validator->fails()) {
                return jsonResponse(false, 400, $validator->errors());
            }

            /*$existMobileNumber = (new CustomerModel)->existMobileNumber($inputs['dial_code'], $inputs['dial_code_iso'], $inputs['mobile_number'], $customer->id);
            if (!empty($existMobileNumber)) {
                $message = __('message.mobile_number_already_exists');
                return jsonResponse(false, 207, $message);
            }*/



            $data = $request->except(['image']);

            if ($request->hasFile('image')) {
                $image =  uploadFile('image', CUSTOMER_PATH, $customer->image);
                $data['image'] = $image;
            }
            //password check if null
            if (!empty($request->password)) {
                $data['password'] = Hash::make($request->password);
            } else {
                $data['password'] = $customer->password;
            }

            //invalid number

            if (!isValidNumber($request->dial_code, $request->dial_code_iso, $request->mobile_number)) {
                $message = __('message.invalid_mobile_number');
                return jsonResponse(false, 207, $message);
            }

            //check status and logged out
            /*if ($customer->status == INACTIVE) {
                $customer->tokens()->delete();
            }*/
            if ($request->status == ACTIVE) {
                $customer->last_reactivated_at = now();
            }

            $customer->update($data);
            DB::commit();
            $message = __('message.customer_updated_successfully');
            $extra = ['redirect' =>  route('admin.customer.index')];
            return jsonResponse(true, 200, $message, $extra);
        } catch (\Exception $e) {
            DB::rollBack();
            $message = __('message.internal_server_error') . $e->getMessage();
            return jsonResponse(false, 500, $message);
        }
    }



    public function destroy(CustomerModel $customer)
    {
        if (!empty($customer)) {
            if ($customer->image && file_exists(public_path('uploads/customers/' . $customer->image))) {
                unlink(public_path('uploads/customers/' . $customer->image));
            }
            try {
                DB::beginTransaction();
                $customer->delete();
                $customer->tokens()->delete();
                DB::commit();
                $message = __('message.customer_deleted_successfully');
                $extra = ['redirect' => route('admin.customer.index')];
                return jsonResponse(true, 200, $message, $extra);
            } catch (\Exception $e) {
                DB::rollBack();
                $message = __('message.internal_server_error') . $e->getMessage();
                $extra = ['redirect' => route('admin.customer.index')];
                return jsonResponse(false, 500, $message, $extra);
            }

            // return redirect()->route('admin.customer')->with('success', 'Customer deleted successfully.');
        } else {
            $message = __('message.invalid_details');
            $extra = ['redirect' => route('admin.customer.index')];
            return jsonResponse(false, 500, $message, $extra);
        }
    }
    public function getStates($countryId)
    {
        $states = DB::table('states')->where('country_id', $countryId)->select('id', 'name')->orderBy('name')->get();
        return response()->json(
            $states
        );
    }
    public function getCities($stateId)
    {
        $cities = DB::table('cities')->where('state_id', $stateId)->select('id', 'name')->orderBy('name')->get();
        //dd($stateId);
        return response()->json(
            $cities
        );
    }
    public function sendMessage(Request $request)
    {
        $mobileNumber = $request->dial_code . $request->mobile_number;
        $message = $request->message;
        // send SMS via SMSService
        app(SMSService::class)->sendNote($mobileNumber, $message);
        return jsonResponse(true, 200, __('message.message_sent'));
    }
}
